<?php

namespace App\Models;

use GrahamCampbell\Markdown\Facades\Markdown;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;

class Post extends Model
{
    use HasFactory;
    use HasSlug;

    protected $table = 'post';
    protected $primaryKey = 'post_id';

    protected $fillable = [
        'title',
        'slug',
        'thumbnail',
        'content',
        'rendered_content',
        'tags',
        'user_id',
        'view_count'
    ];

    public function scopeSlug($query, $slug) {
        return $query->where('slug', $slug);
    }

    public function views() {
        return $this->hasMany(PostView::class, 'post_id', 'post_id');
    }


    public function getRouteKeyName()
    {
        return 'slug';
    }


    public function getHeadlineAttribute() {
        $c = strip_tags($this->content);
        return substr($c, 0, 100) . (strlen($c) > 100 ? '...' : '');
    }


    public static function boot() {
        parent::boot();

        self::creating(function($model) {
            $user = auth()->user();
            if($user == null) {
                $user = User::first();
            }
            $model->user_id = $user->user_id;
            $model->tags = json_encode($model->tags);
        });
    }


    public function getSlugOptions() : SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('title')
            ->saveSlugsTo('slug');
    }


    public function getTagsOptionAttribute() {
        $option = json_decode($this->tags, true);
        $result = [];
        foreach($option as $k => $value) {
            $result[$value] = $value;
        }
        return $result;
    }

    public function user() {
        return $this->belongsTo(User::class, 'user_id', 'user_id');
    }
}
